/*
 * File: XKCDColorWheel.java
 * ========================================================
 * A visualizer that can be used to display colors based on
 * their hue (what color) and saturation (how intense).
 * 
 * This visualizer draws a circle in the window for plotting
 * colors.  The radius is given by the intensity of the color,
 * and the angle from the center is chosen by the hue of the
 * color.
 */
import java.awt.*;
import acm.graphics.*;
import acm.util.*;

public class XKCDColorWheel extends GCanvas {
	/* A constant detailing what fraction of the maximum possible radius should be
	 * used.
	 */
	private static final double RADIUS_MODIFIER = 0.95;
	
	/**
	 * Clears the display of all points.
	 */
	public void clear() {
		removeAll();
		addCircle();
	}
	
	/**
	 * Given an RGB triplet, plots that triplet based on its hue and saturation.
	 * 
	 * @param triplet An array of the RGB values.
	 */
	public void plotColor(int[] triplet) {
		/* Validate the input. */
		if (triplet.length != 3)
			throw new ErrorException("Input has the wrong length.");
		
		/* Convert the color to an HSB triplet of hue (color), saturation
		 * (intensity), and brightness.
		 */
		float[] components = Color.RGBtoHSB(triplet[0], triplet[1], triplet[2], null);
		
		/* The radius is given by the intensity of the color, which is in slot 1. */
		double radius = getRadius() * components[1];
		
		/* The angle is given by the hue of the color, which is in slot 0. */
		double theta = components[0] * Math.PI * 2.0;
		
		/* Determine the x and y coordinates. */
		double x = getWidth() / 2.0  + radius * Math.cos(theta);
		double y = getHeight() / 2.0 - radius * Math.sin(theta);
		
		/* Plot a pixel there. */
		GRect pt = new GRect(x, y, 1, 1);
		pt.setFilled(true);
		pt.setColor(new Color(triplet[0], triplet[1], triplet[2]));
		add(pt);
	}
	

	/**
	 * Returns the radius of the outermost circle, which is just slightly less than
	 * half the width of the canvas.
	 * @return
	 */
	private double getRadius() {
		return getWidth() / 2.0 * RADIUS_MODIFIER;
	}

	/**
	 * Draws the outline of the full circle.
	 */
	private void addCircle() {
		double r = getRadius();
		add(new GOval(getWidth() / 2.0 - r, getHeight() / 2.0 - r, 2 * r, 2 * r));
	}
}
